<?php
/**
 * Template: Author Archive (author.php)
 * Focus: E-E-A-T, JSON-LD (Person/BreadcrumbList + BlogPosting per item), accessibility, internal links, RSS, pagination
 */

get_header();

// Author object & meta
$author      = get_queried_object();
$author_id   = isset($author->ID) ? (int) $author->ID : 0;
$displayName = get_the_author_meta('display_name', $author_id);
$description = get_the_author_meta('description', $author_id);

$job_title   = get_user_meta($author_id, 'job_title', true);
$org         = get_user_meta($author_id, 'organization', true);
$website     = get_the_author_meta('user_url', $author_id);

// Social profiles (used for sameAs + visible links)
$same_as_raw = array(
    get_user_meta($author_id, 'twitter', true),
    get_user_meta($author_id, 'linkedin', true),
    get_user_meta($author_id, 'github', true),
    get_user_meta($author_id, 'instagram', true),
    get_user_meta($author_id, 'facebook', true),
);
$same_as = array_values(array_filter(array_map('esc_url_raw', $same_as_raw)));

$post_count = (int) count_user_posts($author_id, 'post', true);
$avatar_url = get_avatar_url($author_id, array('size' => 256, 'default' => 'blank'));

// Latest post (for "Last updated" signal)
$latest_post_ids = get_posts(array(
    'author'         => $author_id,
    'posts_per_page' => 1,
    'orderby'        => 'date',
    'order'          => 'DESC',
    'post_status'    => 'publish',
    'fields'         => 'ids',
));
$latest_post_id        = $latest_post_ids[0] ?? 0;
$latest_modified_iso   = $latest_post_id ? get_post_modified_time('c', true, $latest_post_id) : '';
$latest_modified_human = $latest_post_id ? get_post_modified_time(get_option('date_format'), true, $latest_post_id) : '';

// JSON-LD: Person
$person_ld = array_filter(array(
    '@context'    => 'https://schema.org',
    '@type'       => 'Person',
    'name'        => $displayName,
    'description' => $description ? wp_strip_all_tags($description) : null,
    'image'       => $avatar_url ?: null,
    'jobTitle'    => $job_title ?: null,
    'worksFor'    => $org ? array('@type' => 'Organization', 'name' => $org) : null,
    'url'         => $website ?: home_url('/'),
    'sameAs'      => !empty($same_as) ? $same_as : null,
));

// JSON-LD: BreadcrumbList
$breadcrumbs_ld = array(
    '@context'        => 'https://schema.org',
    '@type'           => 'BreadcrumbList',
    'itemListElement' => array(
        array(
            '@type'    => 'ListItem',
            'position' => 1,
            'name'     => get_bloginfo('name'),
            'item'     => home_url('/'),
        ),
        array(
            '@type'    => 'ListItem',
            'position' => 2,
            'name'     => $displayName,
            'item'     => get_author_posts_url($author_id),
        ),
    ),
);
?>
<!-- Structured Data -->
<script type="application/ld+json"><?php echo wp_json_encode($person_ld, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES); ?></script>
<script type="application/ld+json"><?php echo wp_json_encode($breadcrumbs_ld, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES); ?></script>

<main id="main" class="container" itemscope itemtype="https://schema.org/ProfilePage">
    <!-- Breadcrumb for UX -->
    <nav class="breadcrumb" aria-label="breadcrumb" style="margin:1rem 0;">
        <a href="<?php echo esc_url(home_url('/')); ?>">Home</a>
        <span aria-hidden="true"> » </span>
        <span><?php echo esc_html($displayName); ?></span>
    </nav>

    <div class="cta" itemprop="mainEntity" itemscope itemtype="https://schema.org/Person">
        <h1 class="post-title" itemprop="name"><?php echo esc_html($displayName); ?></h1>

        <?php if ($job_title || $org): ?>
            <p class="author-job" style="margin:.25rem 0 .75rem;">
                <?php
                $job_line = trim(($job_title ?: '') . (($job_title && $org) ? ', ' : '') . ($org ?: ''));
                echo esc_html($job_line);
                ?>
            </p>
        <?php endif; ?>

        <div class="author-card" style="display:flex;gap:1rem;align-items:flex-start;">
            <div class="author-avatar">
                <?php echo get_avatar($author_id, 128, 'blank', $displayName, array(
                    'class'        => 'rounded',
                    'loading'      => 'lazy',
                    'extra_attr'   => 'decoding="async" fetchpriority="low"'
                )); ?>
            </div>

            <div class="author-info" style="flex:1;">
                <?php if (!empty($description)): ?>
                    <div class="author-bio" itemprop="description">
                        <?php echo wpautop(wp_kses_post($description)); ?>
                    </div>
                <?php else: ?>
                    <p class="muted">This author’s bio hasn’t been added yet.</p>
                <?php endif; ?>
                <p class="author-stats" style="margin-top:.5rem;">
                    <?php if ($post_count > 0): ?>
                        <strong>Posts:</strong> <?php echo (int) $post_count; ?>
                        <?php if ($latest_modified_iso): ?>
                            • <strong>Last updated:</strong>
                            <time datetime="<?php echo esc_attr($latest_modified_iso); ?>">
                                <?php echo esc_html($latest_modified_human); ?>
                            </time>
                        <?php endif; ?>
                    <?php else: ?>
                        No published posts yet.
                    <?php endif; ?>
                </p>
            </div>
        </div>
    </div>

    <?php if (have_posts()): ?>
        <div class="screenshots" role="list">
            <?php while (have_posts()): the_post(); ?>
                <article class="screenshot" role="listitem" itemscope itemtype="https://schema.org/BlogPosting">
                    <a href="<?php the_permalink(); ?>" itemprop="url">
                        <div class="post-thumbnail" itemprop="image" itemscope itemtype="https://schema.org/ImageObject">
                            <?php
                            if (has_post_thumbnail()) {
                                the_post_thumbnail('medium', array(
                                    'loading'        => 'lazy',
                                    'alt'            => esc_attr(get_the_title()),
                                    'decoding'       => 'async',
                                    'fetchpriority'  => 'low'
                                ));
                            } else {
                                echo '<img src="' . esc_url(get_template_directory_uri()) . '/images/placeholder.png" alt="No image available" loading="lazy" width="300" height="200" decoding="async">';
                            }
                            ?>
                        </div>

                        <h3 itemprop="headline"><?php the_title(); ?></h3>
                        <p itemprop="description"><?php echo esc_html(wp_trim_words(get_the_excerpt(), 20)); ?></p>
                    </a>

                    <!-- Essential article metadata -->
                    <meta itemprop="datePublished" content="<?php echo esc_attr(get_the_date('c')); ?>">
                    <meta itemprop="dateModified" content="<?php echo esc_attr(get_the_modified_date('c')); ?>">
                    <meta itemprop="author" content="<?php echo esc_attr($displayName); ?>">
                    <?php if ($avatar_url): ?>
                        <meta itemprop="author.url" content="<?php echo esc_url(get_author_posts_url($author_id)); ?>">
                        <meta itemprop="author.image" content="<?php echo esc_url($avatar_url); ?>">
                    <?php endif; ?>
                </article>
            <?php endwhile; ?>
        </div>

        <div class="post-meta" style="margin-top:1.25rem;">
            <?php
            the_posts_pagination(array(
                'prev_text'           => '« Previous',
                'next_text'           => 'Next »',
                'screen_reader_text'  => 'Posts by ' . $displayName,
                'mid_size'            => 1,
            ));
            ?>
        </div>
    <?php else: ?>
        <p>No posts found for this author.</p>
    <?php endif; ?>

    <?php
    // Helpful internal links (if these pages exist)
    $about_page   = get_page_by_path('about');
    $contact_page = get_page_by_path('contact');
    if ($about_page || $contact_page): ?>
        <div class="related-links" style="margin-top:1.5rem;">
            <h2 class="h5">More</h2>
            <ul>
                <?php if ($about_page): ?>
                    <li><a href="<?php echo esc_url(get_permalink($about_page->ID)); ?>">About</a></li>
                <?php endif; ?>
                <?php if ($contact_page): ?>
                    <li><a href="<?php echo esc_url(get_permalink($contact_page->ID)); ?>">Contact</a></li>
                <?php endif; ?>
            </ul>
        </div>
    <?php endif; ?>
</main>

<?php get_footer(); ?>
